package com.acme.fileserver.controller;

import com.acme.fileserver.model.Session;
import com.acme.fileserver.model.User;
import com.acme.fileserver.repository.SessionRepository;
import com.acme.fileserver.repository.UserRepository;
import com.acme.fileserver.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Controller
@RequestMapping("/users")
public class UserController {
    
    @Autowired
    private UserService userService;
    
    @Autowired
    private SessionRepository sessionRepository;
    
    @Autowired
    private UserRepository userRepository;
    
    private Long getCurrentUserId(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals("sessionId")) {
                    Optional<Session> sessionOpt = sessionRepository.findBySessionId(cookie.getValue());
                    if (sessionOpt.isPresent() && !sessionOpt.get().isExpired()) {
                        return sessionOpt.get().getUserId();
                    }
                }
            }
        }
        return null;
    }
    
    private boolean isAdmin(HttpServletRequest request) {
        Long userId = getCurrentUserId(request);
        if (userId != null) {
            Optional<User> userOpt = userRepository.findById(userId);
            if (userOpt.isPresent()) {
                return "ADMIN".equals(userOpt.get().getRole());
            }
        }
        return false;
    }
    
    @GetMapping("/profile")
    public String profilePage(HttpServletRequest request, Model model) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        Optional<User> userOpt = userService.getUserById(userId);
        if (userOpt.isPresent()) {
            model.addAttribute("user", userOpt.get());
        }
        
        return "profile";
    }
    
    @PostMapping("/profile")
    public String updateProfile(@RequestParam String email,
                               @RequestParam String fullName,
                               HttpServletRequest request,
                               RedirectAttributes redirectAttributes) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        // VULN: Stored XSS - user input not sanitized
        // VULN: Insecure Direct Object Reference - no authorization check
        User updated = userService.updateUser(userId, email, fullName);
        
        if (updated != null) {
            redirectAttributes.addFlashAttribute("success", "Profile updated");
        } else {
            redirectAttributes.addFlashAttribute("error", "Update failed");
        }
        
        return "redirect:/users/profile";
    }
    
    @PostMapping("/password")
    public String updatePassword(@RequestParam String newPassword,
                                HttpServletRequest request,
                                RedirectAttributes redirectAttributes) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        // VULN: Weak password update - no old password verification
        boolean updated = userService.updatePassword(userId, newPassword);
        
        if (updated) {
            redirectAttributes.addFlashAttribute("success", "Password updated");
        } else {
            redirectAttributes.addFlashAttribute("error", "Password update failed");
        }
        
        return "redirect:/users/profile";
    }
    
    @GetMapping("/list")
    public String listUsers(@RequestParam(required = false) String search,
                           HttpServletRequest request,
                           Model model) {
        // VULN: Missing Authorization - should check admin role
        // Currently allows any authenticated user to see all users
        
        List<User> users;
        if (search != null && !search.isEmpty()) {
            // VULN: SQL Injection - unsafe search
            users = userService.searchUsersUnsafe(search);
        } else {
            users = userService.getAllUsers();
        }
        
        // VULN: Sensitive Data Exposure - displaying password hashes
        model.addAttribute("users", users);
        model.addAttribute("searchTerm", search);
        return "users";
    }
    
    // REST API endpoints
    @GetMapping("/api/{userId}")
    @ResponseBody
    public Map<String, Object> getUserApi(@PathVariable Long userId,
                                          HttpServletRequest request) {
        Map<String, Object> response = new HashMap<>();
        
        // VULN: Insecure Direct Object Reference - no authorization
        Optional<User> userOpt = userService.getUserById(userId);
        if (userOpt.isPresent()) {
            User user = userOpt.get();
            response.put("id", user.getId());
            response.put("username", user.getUsername());
            response.put("email", user.getEmail());
            response.put("fullName", user.getFullName());
            // VULN: Sensitive Data Exposure - returning password hash
            response.put("password", user.getPassword());
            response.put("role", user.getRole());
        } else {
            response.put("error", "User not found");
        }
        
        return response;
    }
    
    @PostMapping("/api/update")
    @ResponseBody
    public Map<String, Object> updateUserApi(@RequestBody User userData,
                                             HttpServletRequest request) {
        Map<String, Object> response = new HashMap<>();
        
        // VULN: Mass Assignment - accepting all fields from request
        // VULN: CSRF - no token validation
        User updated = userService.updateUserUnsafe(userData);
        
        if (updated != null) {
            response.put("success", true);
            response.put("user", updated);
        } else {
            response.put("success", false);
            response.put("error", "Update failed");
        }
        
        return response;
    }
}
