package com.acme.fileserver.controller;

import com.acme.fileserver.model.FileEntity;
import com.acme.fileserver.model.Session;
import com.acme.fileserver.model.User;
import com.acme.fileserver.repository.SessionRepository;
import com.acme.fileserver.repository.UserRepository;
import com.acme.fileserver.service.FileService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Controller
@RequestMapping("/files")
public class FileController {
    
    @Autowired
    private FileService fileService;
    
    @Autowired
    private SessionRepository sessionRepository;
    
    @Autowired
    private UserRepository userRepository;
    
    private Long getCurrentUserId(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals("sessionId")) {
                    Optional<Session> sessionOpt = sessionRepository.findBySessionId(cookie.getValue());
                    if (sessionOpt.isPresent() && !sessionOpt.get().isExpired()) {
                        return sessionOpt.get().getUserId();
                    }
                }
            }
        }
        return null;
    }
    
    @GetMapping
    public String listFiles(@RequestParam(required = false) String search,
                           HttpServletRequest request,
                           Model model) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        List<FileEntity> files = fileService.getUserFiles(userId);
        
        // VULN: Stored XSS - search parameter stored and displayed without escaping
        if (search != null && !search.isEmpty()) {
            model.addAttribute("searchTerm", search);
            // Filter files by search term (vulnerable to XSS)
        }
        
        model.addAttribute("files", files);
        return "files";
    }
    
    @GetMapping("/upload")
    public String uploadPage() {
        return "upload";
    }
    
    @PostMapping("/upload")
    public String uploadFile(@RequestParam("file") MultipartFile file,
                            @RequestParam(required = false) String description,
                            HttpServletRequest request,
                            RedirectAttributes redirectAttributes) {
        
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        // VULN: Stored XSS - description not sanitized
        try {
            fileService.uploadFile(file, userId, description);
            redirectAttributes.addFlashAttribute("success", "File uploaded successfully");
        } catch (Exception e) {
            redirectAttributes.addFlashAttribute("error", "Upload failed: " + e.getMessage());
        }
        
        return "redirect:/files";
    }
    
    @GetMapping("/download/{fileId}")
    public ResponseEntity<Resource> downloadFile(@PathVariable Long fileId,
                                                HttpServletRequest request) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return ResponseEntity.status(401).build();
        }
        
        // VULN: Insecure Direct Object Reference - no authorization check
        File file = fileService.downloadFile(fileId, userId);
        
        if (file != null && file.exists()) {
            Resource resource = new FileSystemResource(file);
            return ResponseEntity.ok()
                    .contentType(MediaType.APPLICATION_OCTET_STREAM)
                    .header(HttpHeaders.CONTENT_DISPOSITION, 
                           "attachment; filename=\"" + file.getName() + "\"")
                    .body(resource);
        }
        
        return ResponseEntity.notFound().build();
    }
    
    // VULN: Path Traversal endpoint
    @GetMapping("/download/path")
    public ResponseEntity<Resource> downloadFileByPath(@RequestParam String path,
                                                      HttpServletRequest request) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return ResponseEntity.status(401).build();
        }
        
        // VULN: Path Traversal - direct use of user-provided path
        File file = fileService.downloadFileByPath(path);
        
        if (file != null && file.exists()) {
            Resource resource = new FileSystemResource(file);
            return ResponseEntity.ok()
                    .contentType(MediaType.APPLICATION_OCTET_STREAM)
                    .header(HttpHeaders.CONTENT_DISPOSITION, 
                           "attachment; filename=\"" + file.getName() + "\"")
                    .body(resource);
        }
        
        return ResponseEntity.notFound().build();
    }
    
    @PostMapping("/delete/{fileId}")
    public String deleteFile(@PathVariable Long fileId,
                           HttpServletRequest request,
                           RedirectAttributes redirectAttributes) {
        Long userId = getCurrentUserId(request);
        if (userId == null) {
            return "redirect:/auth/login";
        }
        
        // VULN: CSRF - no token validation
        // VULN: Insecure Direct Object Reference
        boolean deleted = fileService.deleteFile(fileId, userId);
        
        if (deleted) {
            redirectAttributes.addFlashAttribute("success", "File deleted successfully");
        } else {
            redirectAttributes.addFlashAttribute("error", "Failed to delete file");
        }
        
        return "redirect:/files";
    }
    
    // REST API endpoints
    @GetMapping("/api/list")
    @ResponseBody
    public Map<String, Object> apiListFiles(HttpServletRequest request) {
        Map<String, Object> response = new HashMap<>();
        Long userId = getCurrentUserId(request);
        
        if (userId == null) {
            response.put("error", "Unauthorized");
            return response;
        }
        
        List<FileEntity> files = fileService.getUserFiles(userId);
        response.put("files", files);
        return response;
    }
    
    // VULN: Command Injection endpoint
    @GetMapping("/api/info")
    @ResponseBody
    public Map<String, String> getFileInfo(@RequestParam String filename,
                                          HttpServletRequest request) {
        Map<String, String> response = new HashMap<>();
        Long userId = getCurrentUserId(request);
        
        if (userId == null) {
            response.put("error", "Unauthorized");
            return response;
        }
        
        // VULN: Command Injection - user input in system command
        String info = fileService.getFileInfo(filename);
        response.put("info", info);
        return response;
    }
}
