package com.acme.fileserver.controller;

import com.acme.fileserver.model.Session;
import com.acme.fileserver.model.User;
import com.acme.fileserver.service.AuthService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.Map;

@Controller
@RequestMapping("/auth")
public class AuthController {
    
    @Autowired
    private AuthService authService;
    
    @GetMapping("/login")
    public String loginPage(@RequestParam(required = false) String error, Model model) {
        // VULN: Reflected XSS - error message not escaped
        if (error != null) {
            model.addAttribute("error", error);
        }
        return "login";
    }
    
    @PostMapping("/login")
    public String login(@RequestParam String username, 
                       @RequestParam String password,
                       @RequestParam(required = false) String redirect,
                       HttpServletRequest request,
                       HttpServletResponse response,
                       RedirectAttributes redirectAttributes) {
        
        User user = authService.authenticate(username, password);
        
        if (user != null && user.isActive()) {
            Session session = authService.createSession(user, request);
            
            // VULN: Session Fixation - not regenerating session ID
            // VULN: Insecure cookie settings
            Cookie cookie = new Cookie("sessionId", session.getSessionId());
            cookie.setHttpOnly(false); // VULN: XSS vulnerability
            cookie.setSecure(false);   // VULN: Transmitted over HTTP
            cookie.setPath("/");
            cookie.setMaxAge(30 * 60); // 30 minutes
            response.addCookie(cookie);
            
            // VULN: Open Redirect - redirect parameter not validated
            if (redirect != null && !redirect.isEmpty()) {
                return "redirect:" + redirect;
            }
            
            return "redirect:/files";
        } else {
            redirectAttributes.addAttribute("error", "Invalid username or password");
            return "redirect:/auth/login";
        }
    }
    
    @GetMapping("/register")
    public String registerPage() {
        return "register";
    }
    
    @PostMapping("/register")
    public String register(@RequestParam String username,
                          @RequestParam String password,
                          @RequestParam String email,
                          RedirectAttributes redirectAttributes) {
        
        // VULN: No input validation
        // VULN: No rate limiting
        try {
            User user = authService.registerUser(username, password, email);
            redirectAttributes.addAttribute("success", "Registration successful");
            return "redirect:/auth/login";
        } catch (Exception e) {
            redirectAttributes.addAttribute("error", "Registration failed: " + e.getMessage());
            return "redirect:/auth/register";
        }
    }
    
    @GetMapping("/logout")
    public String logout(HttpServletRequest request, HttpServletResponse response) {
        Cookie[] cookies = request.getCookies();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals("sessionId")) {
                    authService.logout(cookie.getValue());
                    cookie.setMaxAge(0);
                    response.addCookie(cookie);
                }
            }
        }
        return "redirect:/auth/login";
    }
    
    // REST API endpoint
    @PostMapping("/api/login")
    @ResponseBody
    public Map<String, Object> apiLogin(@RequestParam String username,
                                        @RequestParam String password,
                                        HttpServletRequest request) {
        Map<String, Object> response = new HashMap<>();
        
        User user = authService.authenticate(username, password);
        if (user != null) {
            Session session = authService.createSession(user, request);
            response.put("success", true);
            response.put("sessionId", session.getSessionId());
            // VULN: Sensitive Data Exposure - returning password hash
            response.put("user", user);
        } else {
            response.put("success", false);
            response.put("message", "Invalid credentials");
        }
        
        return response;
    }
}
