# ACME File Server

A modern, feature-rich file server application built with Spring Boot. ACME File Server provides secure file storage, user management, and a RESTful API for programmatic access to your files.

## Features

- **User Authentication & Authorization**: Secure user registration and login system with session management
- **File Management**: Upload, download, and organize files with metadata support
- **User Profiles**: Manage user information and preferences
- **File Search**: Quick search functionality to find files by name or description
- **REST API**: Comprehensive RESTful API for integration with other applications
- **Web Interface**: Clean, intuitive web interface built with Thymeleaf
- **Database Support**: Flexible database support with H2 (development) and MySQL (production)

## Technology Stack

- **Framework**: Spring Boot 2.7.18
- **Database**: H2 (in-memory) / MySQL
- **Template Engine**: Thymeleaf
- **Build Tool**: Maven
- **Java Version**: 11+

## Quick Start

### Prerequisites

- Java 11 or higher
- Maven 3.6 or higher
- (Optional) MySQL 8.0+ for production deployments

### Installation

1. Clone the repository:
```bash
git clone https://github.com/acme/fileserver.git
cd fileserver
```

2. Build the application:
```bash
mvn clean install
```

3. Run the application:
```bash
mvn spring-boot:run
```

4. Access the application:
   - Web interface: http://localhost:8080
   - H2 Console (development): http://localhost:8080/h2-console
     - JDBC URL: `jdbc:h2:mem:fileserverdb`
     - Username: `sa`
     - Password: (leave empty)

### First Steps

1. Navigate to http://localhost:8080
2. Click "Register new account" to create your first user account
3. Log in with your credentials
4. Start uploading and managing your files!

## Configuration

### Application Properties

The application can be configured via `src/main/resources/application.properties`:

```properties
# Server Configuration
server.port=8080

# Database Configuration (H2 for development)
spring.datasource.url=jdbc:h2:mem:fileserverdb
spring.datasource.username=sa
spring.datasource.password=

# File Upload Configuration
spring.servlet.multipart.max-file-size=50MB
spring.servlet.multipart.max-request-size=50MB

# Session Configuration
server.servlet.session.timeout=30m
```

### Production Configuration

For production deployments, configure MySQL:

```properties
spring.datasource.url=jdbc:mysql://localhost:3306/fileserver
spring.datasource.username=your_username
spring.datasource.password=your_password
spring.jpa.hibernate.ddl-auto=update
```

## API Documentation

### Authentication

#### Register User
```http
POST /auth/register
Content-Type: application/x-www-form-urlencoded

username=john&password=securepass&email=john@example.com
```

#### Login
```http
POST /auth/login
Content-Type: application/x-www-form-urlencoded

username=john&password=securepass
```

#### API Login (JSON)
```http
POST /auth/api/login
Content-Type: application/x-www-form-urlencoded

username=john&password=securepass
```

Response:
```json
{
  "success": true,
  "sessionId": "session_token_here",
  "user": {
    "id": 1,
    "username": "john",
    "email": "john@example.com"
  }
}
```

### File Operations

#### List Files
```http
GET /files/api/list
Cookie: sessionId=your_session_token
```

#### Upload File
```http
POST /files/upload
Content-Type: multipart/form-data
Cookie: sessionId=your_session_token

file=<file_data>&description=Optional file description
```

#### Download File
```http
GET /files/download/{fileId}
Cookie: sessionId=your_session_token
```

#### Download File by Path
```http
GET /files/download/path?path=relative/path/to/file.txt
Cookie: sessionId=your_session_token
```

#### Delete File
```http
POST /files/delete/{fileId}
Cookie: sessionId=your_session_token
```

#### Get File Information
```http
GET /files/api/info?filename=example.txt
Cookie: sessionId=your_session_token
```

### User Management

#### Get User Profile
```http
GET /users/api/{userId}
Cookie: sessionId=your_session_token
```

#### Update User Profile
```http
POST /users/api/update
Content-Type: application/json
Cookie: sessionId=your_session_token

{
  "id": 1,
  "email": "newemail@example.com",
  "fullName": "John Doe"
}
```

#### List All Users
```http
GET /users/list?search=search_term
Cookie: sessionId=your_session_token
```

### Utility Endpoints

#### Process XML
```http
POST /api/xml/process
Content-Type: application/json

"<?xml version='1.0'?><root>data</root>"
```

#### Fetch URL
```http
GET /api/fetch?url=https://example.com
Cookie: sessionId=your_session_token
```

#### Deserialize Data
```http
POST /api/deserialize
Content-Type: application/json

{
  "data": "base64_encoded_serialized_data"
}
```

## Project Structure

```
src/
├── main/
│   ├── java/com/acme/fileserver/
│   │   ├── controller/          # REST and web controllers
│   │   │   ├── AuthController.java
│   │   │   ├── FileController.java
│   │   │   ├── UserController.java
│   │   │   ├── ApiController.java
│   │   │   └── IndexController.java
│   │   ├── model/                # JPA entity models
│   │   │   ├── User.java
│   │   │   ├── FileEntity.java
│   │   │   └── Session.java
│   │   ├── repository/           # Data access layer
│   │   │   ├── UserRepository.java
│   │   │   ├── FileRepository.java
│   │   │   └── SessionRepository.java
│   │   ├── service/              # Business logic
│   │   │   ├── AuthService.java
│   │   │   ├── FileService.java
│   │   │   └── UserService.java
│   │   └── FileserverApplication.java
│   └── resources/
│       ├── templates/            # Thymeleaf templates
│       │   ├── login.html
│       │   ├── register.html
│       │   ├── files.html
│       │   ├── upload.html
│       │   ├── profile.html
│       │   └── users.html
│       └── application.properties
└── test/                         # Unit and integration tests
```

## Architecture

### Controllers
- **AuthController**: Handles user authentication, registration, and session management
- **FileController**: Manages file upload, download, and deletion operations
- **UserController**: Handles user profile management and user listing
- **ApiController**: Provides utility API endpoints for XML processing and URL fetching
- **IndexController**: Routes root requests to appropriate pages

### Services
- **AuthService**: Authentication logic, password hashing, and session generation
- **FileService**: File storage operations, path management, and file metadata handling
- **UserService**: User data management and search functionality

### Models
- **User**: Represents application users with authentication credentials
- **FileEntity**: Stores file metadata and ownership information
- **Session**: Manages user sessions and authentication tokens

## Usage Examples

### Upload a File via Web Interface

1. Log in to the application
2. Navigate to "Upload File"
3. Select a file and optionally add a description
4. Click "Upload"

### Upload a File via API

```bash
curl -X POST http://localhost:8080/files/upload \
  -H "Cookie: sessionId=your_session_token" \
  -F "file=@/path/to/your/file.txt" \
  -F "description=My important file"
```

### Download a File

```bash
curl -X GET http://localhost:8080/files/download/1 \
  -H "Cookie: sessionId=your_session_token" \
  -o downloaded_file.txt
```

### Search for Files

Navigate to the files page and use the search box, or access:
```
http://localhost:8080/files?search=keyword
```

## Database Schema

### Users Table
- `id`: Primary key
- `username`: Unique username
- `password`: Hashed password
- `email`: User email address
- `fullName`: User's full name
- `role`: User role (USER, ADMIN)
- `createdAt`: Account creation timestamp
- `lastLogin`: Last login timestamp
- `active`: Account status

### Files Table
- `id`: Primary key
- `filename`: Original filename
- `filepath`: Storage path on server
- `contentType`: MIME type
- `fileSize`: File size in bytes
- `ownerId`: Foreign key to users table
- `uploadedAt`: Upload timestamp
- `description`: User-provided description
- `isPublic`: Public visibility flag

### Sessions Table
- `id`: Primary key
- `sessionId`: Unique session identifier
- `userId`: Foreign key to users table
- `createdAt`: Session creation timestamp
- `expiresAt`: Session expiration timestamp
- `ipAddress`: Client IP address
- `userAgent`: Client user agent string

## Development

### Running in Development Mode

The application uses H2 in-memory database by default, which resets on each restart. This is ideal for development and testing.

### Building for Production

```bash
mvn clean package
java -jar target/fileserver-1.0.0.jar
```

### Database Migrations

The application uses JPA auto-ddl for schema management. For production, consider using Flyway or Liquibase for version-controlled migrations.

## Contributing

We welcome contributions! Please follow these steps:

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add some amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

## License

Copyright (c) 2024 ACME Corporation. All rights reserved.

This software is proprietary and confidential. Unauthorized copying, modification, distribution, or use of this software, via any medium, is strictly prohibited.

## Support

For issues, questions, or feature requests, please contact:
- Email: support@acme.com
- Documentation: https://docs.acme.com/fileserver
- Issue Tracker: https://github.com/acme/fileserver/issues

## Changelog

### Version 1.0.0
- Initial release
- User authentication and registration
- File upload and download functionality
- REST API endpoints
- Web interface
- Session management
- User profile management
